<?php

namespace App\Http\Controllers;

use App\Models\Household;
use App\Models\HouseholdAssignment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\HouseholdsImport;
use Illuminate\Support\Carbon;
use App\Models\Segment;
use App\Models\Region;
use App\Models\Departamento; // Asegúrate de importar los modelos correctos
use App\Models\Municipio;
use App\Models\Distrito;
use App\Models\Area;
use Illuminate\Support\Facades\Log;

class HouseholdController extends Controller
{
    public function index()
    {
        $households = Household::all();
        return view('households.index', compact('households'));
    }

     public function create()
    {
        // Puedes cargar los segments si no usas AJAX para el buscador
        $segments = Segment::all();
	$regions = Region::all();

        // Opcional: Cargar departamentos asociados a una región específica (si es necesario)
        // $departamentos = Departamento::where('region', $someRegionCode)->get();

        return view('households.create', compact('segments', 'regions'));
    }

    public function search(Request $request)
    {
        $search = $request->query('q');

        if (is_null($search) || $search === '') {
            return response()->json(['error' => 'Search query is required'], 400);
        }

        // Buscar segmentos donde correlativo_segmento coincida parcialmente con el término de búsqueda
        $segments = Segment::where('correlativo_segmento', 'LIKE', "%$search%")
			   ->where('active', 1)
    			   ->get();

        return response()->json($segments);
    }

    public function store(Request $request)
    {
        $request->validate([
            'correlativo_segmento' => 'required|integer|unique:households,correlativo_segmento',
            'region_de_salud' => 'required|integer',
            'departamento' => 'required|integer',
            'municipio' => 'required|integer',
            'distrito' => 'required|integer',
            'area' => 'required|integer',
            'orden_vivienda' => 'required|integer',
            'numero_vivienda' => 'required|string',
        ]);

        Household::create([
            'correlativo_segmento' => $request->correlativo_segmento,
            'region_de_salud' => $request->region_de_salud,
            'departamento' => $request->departamento,
            'municipio' => $request->municipio,
            'distrito' => $request->distrito,
            'area' => $request->area,
            'orden_vivienda' => $request->orden_vivienda,
            'numero_vivienda' => $request->numero_vivienda,
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now(),
        ]);

        return redirect()->route('households.index')->with('success', 'Household agregado con éxito');
    }

    public function getHouseholdsCount()
    {
        $count = Household::count('id');
        return response()->json(['count' => $count]);
    }
    public function show(Household $household)
    {
        return view('households.show', compact('household'));
    }

    public function edit(Household $household)
    {
        return view('households.edit', compact('household'));
    }

    public function update(Request $request, Household $household)
    {
        $request->validate([
            'correlativo_segmento' => 'required|integer|unique:households,correlativo_segmento,' . $household->id,
            'region_de_salud' => 'required|integer',
            'departamento' => 'required|integer',
            'municipio' => 'required|integer',
            'distrito' => 'required|integer',
            'area' => 'required|integer',
            'orden_vivienda' => 'required|integer',
            'numero_vivienda' => 'required|string',
        ]);

        $household->update([
            'correlativo_segmento' => $request->correlativo_segmento,
            'region_de_salud' => $request->region_de_salud,
            'departamento' => $request->departamento,
            'municipio' => $request->municipio,
            'distrito' => $request->distrito,
            'area' => $request->area,
            'orden_vivienda' => $request->orden_vivienda,
            'numero_vivienda' => $request->numero_vivienda,
            'updated_at' => Carbon::now(),
        ]);

        return redirect()->route('households.index')->with('success', 'Household actualizado con éxito');
    }

    public function destroy(Household $household)
    {
        $household->delete();

        return redirect()->route('households.index')->with('success', 'Household eliminado con éxito');
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:csv,xlsx,xls'
        ]);

        Excel::import(new HouseholdsImport, $request->file('file'));

        return redirect()->route('households.index')->with('success', 'Households importados con éxito');
    }

    public function loadDependentData(Request $request)
    {
        $type = $request->query('type');
        $parentValue = $request->query('parent_value');

        switch ($type) {
            case 'region_de_salud':
                $data = Departamento::where('codigo', $parentValue)->get(['id', 'nombre']);
                break;
            case 'departamento':
                $data = Municipio::where('codigo', $parentValue)->get(['id', 'nombre']);
                break;
            case 'municipio':
                $data = Distrito::where('codigo', $parentValue)->get(['id', 'nombre']);
                break;
            case 'distrito':
                $data = Area::where('codigo', $parentValue)->get(['id', 'nombre']);
                break;
            default:
                $data = [];
                break;
        }

        return response()->json($data);
    }
    public function getDepartamentosByRegion(Request $request)
    {
    $regionCode = $request->query('region');
    $departamentos = Departamento::where('region', $regionCode)->get();

    return response()->json($departamentos);
    }
    public function getSegmentDetails($segmentId)
    {
        // Obtener el segmento y sus hogares relacionados
    $segment = Segment::with(['households'])->findOrFail($segmentId);

    $userId = Auth::user()->id; // Obtener el ID del usuario autenticado

    // Obtener los hogares del segmento
    $households = $segment->households->map(function($household) use ($userId) {
        // Obtener la asignación del hogar
        $assignment = HouseholdAssignment::where('household_id', $household->id)->first();
	
	// Determinar si el hogar está bloqueado
        $isBlocked = $assignment && $assignment->user_id !== $currentUserId;
	
	return [
            'id' => $household->id,
            'orden_vivienda' => $household->orden_vivienda,
            'num_casa' => $household->num_casa,
            'status' => $household->status,
            'assigned_user_id' => $assignment ? $assignment->user_id : null,
            'isBlocked' => $isBlocked, 
        ];
    });

    // Formatear la respuesta
    $data = [
        'region' => $segment->region,
        'department' => $segment->department,
        'muni' => $segment->muni,
        'distrito' => $segment->distrito,
        'area' => $segment->area,
        'households' => $households,
    ];

    return response()->json($data);
    }

    /**
     * Obtener los miembros de un hogar específico.
     */
    public function getHouseholdMembers($householdId)
    {
        // Obtener el hogar y sus miembros relacionados
        $household = Household::with(['members'])->findOrFail($householdId);
        
        // Formatear los datos de los miembros
        $data = [
            'members' => $household->members->map(function($member) {
                return [
		    'numero_de_linea'=> $member->numero_de_linea,
                    'nombre' => $member->nombre,
		    'edad'=> $member->edad,
                    'grupo_edad_texto' => $member->grupo_edad_texto, // Texto del grupo de edad
                    'sexo_texto' => $member->sexo_texto,             // Texto del sexo
                    'seleccionado_texto' => $member->seleccionado_texto, // Texto del campo seleccionado
                ];
            }),
        ];

        return response()->json($data);
    }
    public function getUserId()
    {
        return response()->json(['user_id' => Auth::user()->id]);
    }
    public function assignHousehold(Request $request)
    {
    $validatedData = $request->validate([
        'household_id' => 'required|integer|exists:households,id',
        'user_id' => 'required|integer|exists:users,id',
        'form_id' => 'required|integer',
        'estatus' => 'required|integer',
    ]);

    // Verificar si el hogar ya está completo
    $household = Household::where('id', $validatedData['household_id'])->first();
    if ($household->is_complete) {
        return response()->json(['message' => 'Este hogar ya está marcado como completo.'], 403);
    }
    // Actualizar el estado del hogar a "selected"
    $household->update(['status' => 'selected']);


    // Verificar si el hogar ya está asignado
    $householdAssignment = HouseholdAssignment::where('household_id', $validatedData['household_id'])->first();

    if ($householdAssignment) {
        // Si está asignado a otro usuario
        if ($householdAssignment->user_id != $validatedData['user_id']) {
            return response()->json(['message' => 'Este hogar ya está asignado a otro usuario.'], 403);
        }

        // Si es el mismo usuario, actualiza la asignación
        $householdAssignment->estatus = $validatedData['estatus'];
        $householdAssignment->updated_at = now();
        $householdAssignment->save();

        return response()->json(['message' => 'Asignación actualizada correctamente.'], 200);
    }

    // Crear una nueva asignación
    HouseholdAssignment::create([
        'household_id' => $validatedData['household_id'],
        'user_id' => $validatedData['user_id'],
        'form_id' => $validatedData['form_id'],
        'estatus' => $validatedData['estatus'],
    ]);

    return response()->json(['message' => 'Hogar asignado correctamente.'], 201);
    }


    public function checkAssignment(Request $request){

    	$householdId = $request->input('household_id');
    	$userId = $request->input('user_id');
    	$formId = $request->input('form_id');

    	// Verificar si existe la asignación
    	$assignmentExists = HouseholdAssignment::where('household_id', $householdId)
           ->where('user_id', $userId)
           ->where('form_id', $formId)
           ->where('estatus', 1)
           ->exists();

    	// Devuelve verdadero o falso
    	return response()->json($assignmentExists);
    }
    public function getReplacements($segmentId){

        // Consultar los hogares donde replacement sea igual a 1 y no sea el actual
        $replacements = Household::where('replacement', 1)
                        ->where('correlativo_segmento', '=', $segmentId)
                        ->get(['id', 'orden_vivienda', 'numero_vivienda']); // Traer solo los campos necesarios

        // Retornar los hogares en formato JSON
        return response()->json($replacements);
    }
    
    public function discardHousehold(Request $request)
    {
    // Validar los datos recibidos
    $request->validate([
        'household_id' => 'required|integer',
        'user_id' => 'required|integer',
    ]);

    // Buscar el registro de asignación
    $assignment = HouseholdAssignment::where('household_id', $request->household_id)
                                      ->where('user_id', $request->user_id)
                                      ->where('estatus', 1) // Estatus de "asignado"
                                      ->first();

    if ($assignment) {
        // Cambiar el estatus a 0 (descartado) o eliminar el registro si lo prefieres
        $assignment->estatus = 0;
        $assignment->save();

        return response()->json(['message' => 'Hogar descartado con éxito.']);
    } else {
        return response()->json(['message' => 'No se encontró una asignación activa para descartar.'], 404);
    }
    }
    public function storeSignal(Request $request, $householdId)
{
    $request->validate([
        'signal' => 'required|string'
    ]);

    $household = Household::findOrFail($householdId);
    $household->signal = $request->signal; // Suponiendo que tienes una columna `signal`
    $household->save();

    return response()->json(['message' => 'Señal guardada correctamente']);
    }

}
